/*
/--------------------------------------------------------------------
|
|      bitmap.cpp         Device independent bitmap class
|
|        Manipulates uncompressed device independent bitmaps
|        of all color depths.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdafx.h"
#include "bitmap.h"
//#include "except.h"
//#include "octree.h"

#ifdef _WINDOWS
#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC (CBmp, CObject);
#endif


CBmp::CBmp
    ()
    // Creates an empty bitmap.
    // Derived classes have to create a small bitmap here so the
    // class can assume that a valid bitmap is available all the
    // time.
{
	m_bEmpty = TRUE;
}


CBmp::~CBmp
    ()
{
}


void CBmp::CreateCopy
    ( CBmp & rSrcBmp,
      int BPPWanted
    )
    // Creates a copy of rSrcBmp, converting color depth if nessesary.
    // Supports 1, 8 and 32 BPP. Alpha channel information is preserved.
{
  ASSERT_VALID (this);
  ASSERT_VALID (&rSrcBmp);

  // bdelmee code change
  ASSERT (BPPWanted == 32 || BPPWanted == 8 ||
          BPPWanted == 1 || BPPWanted == 0);
  int BPPSrc = rSrcBmp.GetBitsPerPixel();
  ASSERT (BPPSrc == 32 || BPPSrc == 8 || BPPSrc == 1);

  if (BPPWanted == BPPSrc || BPPWanted == 0)
  {
    if (&rSrcBmp != this)
    {
      // Create empty bitmap.
      Create (rSrcBmp.GetWidth(), rSrcBmp.GetHeight(),
              BPPSrc, rSrcBmp.HasAlpha());
      BYTE ** pSrcLines = rSrcBmp.GetLineArray();
      BYTE ** pDstLines = GetLineArray();
      // Minimum possible line length.
    // bdelmee code change
      int LineLen = GetBytesPerLine();

      for (int y = 0; y<GetHeight(); y++)
        memcpy (pDstLines[y], pSrcLines[y], LineLen);

    // bdelmee code change
      if (GetBitsPerPixel() <= 8)
        SetPalette (rSrcBmp.GetPalette());

      ASSERT_VALID (this);
    }
  }
  else  // bdelmee; bit-depth conversions
  {
    // Can't copy to self while changing bit depth.
    ASSERT (&rSrcBmp != this);

    BOOL bDestAlpha = rSrcBmp.HasAlpha() && BPPWanted != 1;
    Create (rSrcBmp.GetWidth(), rSrcBmp.GetHeight(),
            BPPWanted, bDestAlpha);

    switch (BPPWanted)
    {
      case 32:
        create32BPPCopy (rSrcBmp);
        break;

      case 8:
        //create8BPPCopy (rSrcBmp);
        break;

      case 1:
        create1BPPCopy (rSrcBmp);
        break;
      default:
        ASSERT(FALSE);
      }
    ASSERT_VALID (this);
  }
}

#ifdef _DEBUG
void CBmp::AssertValid
    () const
{
  ASSERT (m_pBits);

  ASSERT (m_Height >= 0);
  ASSERT (m_Width >= 0);

  // Color table only if 8 bpp.
  ASSERT ((m_bpp > 8) == (m_pClrTab == NULL));

  // Alpha channel only if 32 bpp
  ASSERT ((m_bpp == 32) || !m_bAlphaChannel);
}
#endif


/////////////////////////////////////////////////////////////////////
// CBmp creation

void CBmp::Create
    ( LONG Width,
      LONG Height,
      WORD BitsPerPixel,
      BOOL bAlphaChannel
    )
    // Create a new empty bitmap. Bits are uninitialized.
{
  ASSERT_VALID (this);

  m_bEmpty = FALSE;
  freeMembers ();
  internalCreate (Width, Height, BitsPerPixel, bAlphaChannel);

  ASSERT_VALID (this);
}


/////////////////////////////////////////////////////////////////////
// CBmp manipulation

void CBmp::SetGrayPalette
    ()
    // Fills the color table with a grayscale palette.
{
  ASSERT (m_pClrTab); // Bitmap must contain a palette!

  int i;
  int NumColors = GetNumColors();
  double ColFactor = 255/(NumColors-1);

  for (i=0; i<NumColors; i++)
    SetPaletteEntry (i, int(i*ColFactor), int(i*ColFactor), int(i*ColFactor), 0xFF);
}

void CBmp::SetPalette
    ( RGBAPIXEL * pPal
    )
{
  ASSERT (m_pClrTab); // Bitmap must contain a palette!

  // bdelmee code change
  memcpy (m_pClrTab, pPal, GetNumColors() * sizeof(RGBAPIXEL));
}


void CBmp::SetAlphaChannel
    ( CBmp * pAlphaBmp
    )
    // Replaces the alpha channel with a new one.
{
  BYTE * pLine;
  BYTE * pAlphaLine;
  int x,y;
  BYTE ** pLineArray;
  BYTE ** pAlphaLineArray;

  ASSERT_VALID (this);
  // Current bitmap must be 32 bpp.
  ASSERT (GetBitsPerPixel() == 32);

  ASSERT_VALID (pAlphaBmp);
  // Alpha channel must be 8 bpp.
  ASSERT (pAlphaBmp->GetBitsPerPixel() == 8);

  // The two bitmaps must have the same dimensions
  ASSERT (pAlphaBmp->GetWidth() == GetWidth());
  ASSERT (pAlphaBmp->GetHeight() == GetHeight());

  pLineArray = GetLineArray();
  pAlphaLineArray = pAlphaBmp->GetLineArray();

  for (y=0; y < GetHeight(); y++)
  {
    pLine = pLineArray[y];
    pAlphaLine = pAlphaLineArray[y];
    for (x=0; x < GetWidth(); x++)
    {
      pLine[x*4+RGBA_ALPHA] = pAlphaLine[x];
    }
  }

  m_bAlphaChannel = TRUE;

  ASSERT_VALID (this);
}


/////////////////////////////////////////////////////////////////////
// Local functions


void CBmp::initLocals
    ( LONG Width,
      LONG Height,
      WORD BitsPerPixel,
      BOOL bAlphaChannel
    )
{
  m_Width = Width;
  m_Height = Height;
  m_bpp = BitsPerPixel;
  m_bAlphaChannel = bAlphaChannel;

  // Initialize pointers to lines.
  initLineArray ();

  if (BitsPerPixel < 16)
  { // Color table exists
    SetGrayPalette ();
  }

  ASSERT_VALID (this);
}

void CBmp::create32BPPCopy
    ( CBmp & rSrcBmp
    )
{
  int BPPSrc = rSrcBmp.GetBitsPerPixel();
  BYTE ** pSrcLines = rSrcBmp.GetLineArray();
  BYTE ** pDstLines = GetLineArray();
  int SrcLineLen = GetWidth();
  
  if (BPPSrc == 8) // Conversion 8->32 BPP
  {
    RGBAPIXEL *pPal = rSrcBmp.GetPalette();

    for (int y = 0; y<GetHeight(); ++y)
    { // For each line
      BYTE * pSrcPixel = pSrcLines[y];
      BYTE * pDstPixel = pDstLines[y];

      for (int x = 0; x < SrcLineLen; ++x)
      { // For each pixel
        *((RGBAPIXEL *)pDstPixel) = pPal[*pSrcPixel];
        ++pSrcPixel;
        pDstPixel += sizeof(RGBAPIXEL);
      }
    }
  }
  else // 1 -> 32
  {
    RGBAPIXEL *pPal = rSrcBmp.GetPalette();
    RGBAPIXEL blackDot, whiteDot;
    // if bi-tonal "palette" exists, use it...
    if (pPal)
    {
      whiteDot = pPal[0];
      blackDot = pPal[1];
    }
    else
    {
      BYTE *pWhite = (BYTE *) &whiteDot;
      BYTE *pBlack = (BYTE *) &blackDot;
      pWhite[RGBA_RED] = pWhite[RGBA_GREEN] = pWhite[RGBA_BLUE] = 255;
      pBlack[RGBA_RED] = pBlack[RGBA_GREEN] = pBlack[RGBA_BLUE] = 0;
      pWhite[RGBA_ALPHA] = pBlack[RGBA_ALPHA] = 255;
    }

    // assume msb is leftmost
    for (int y = 0; y<GetHeight(); ++y)
    { // For each line
      BYTE * pSrcPixel = pSrcLines[y];
      BYTE * pDstPixel = pDstLines[y];

      for (int x = 0; x < SrcLineLen; ++x)
      { // For each pixel
        if (pSrcPixel[x / 8] & (128 >> (x & 7)))  // black pixel
        *((RGBAPIXEL *)pDstPixel) = blackDot;
        else
        *((RGBAPIXEL *)pDstPixel) = whiteDot;
        pDstPixel += sizeof(RGBAPIXEL);
      }
    }
  }
}

/*
void CBmp::create8BPPCopy
    ( CBmp & rSrcBmp
    )
{
  int BPPSrc = rSrcBmp.GetBitsPerPixel();
  BYTE ** pSrcLines = rSrcBmp.GetLineArray();
  BYTE ** pDstLines = GetLineArray();
  int SrcLineLen = GetWidth();

  if (BPPSrc == 32) // Conversion 32->8 BPP
      quantize (rSrcBmp);
  else // 1 -> 8
  {
    RGBAPIXEL *pPal = rSrcBmp.GetPalette();
    // if bi-tonal "palette" exists, use it...
    if (pPal)
    {
      BYTE *pWhite = (BYTE *) pPal;
      BYTE *pBlack = (BYTE *) (pPal+1);
      SetPaletteEntry(0,
        pWhite[RGBA_RED],pWhite[RGBA_GREEN],pWhite[RGBA_BLUE],
        255);
      SetPaletteEntry(1,
        pBlack[RGBA_RED],pBlack[RGBA_GREEN],pBlack[RGBA_BLUE],
        255);
    }
    else
    {
      SetPaletteEntry(0,255,255,255,255);
      SetPaletteEntry(1,0,0,0,255);
    }

    // assume msb is leftmost
    for (int y = 0; y<GetHeight(); ++y)
    { // For each line
      BYTE * pSrcPixel = pSrcLines[y];
      BYTE * pDstPixel = pDstLines[y];

      for (int x = 0; x < SrcLineLen; ++x)  // For each pixel
        pDstPixel[x] = pSrcPixel[x / 8] & (128 >> (x & 7)) ? 1 : 0;
    }
  }
}
*/
void CBmp::create1BPPCopy
    ( CBmp & rSrcBmp
    )
{
  int BPPSrc = rSrcBmp.GetBitsPerPixel();
  BYTE ** pSrcLines = rSrcBmp.GetLineArray();
  BYTE ** pDstLines = GetLineArray();
  int SrcLineLen = GetWidth();

  SetPaletteEntry(0,255,255,255,255);
  SetPaletteEntry(1,0,0,0,255);

  // downgrade to monochrome
  RGBAPIXEL *pPal = rSrcBmp.GetPalette();
  BYTE *pRGBA;
  int DstLineLen = GetBytesPerLine();

  for (int y = 0; y < GetHeight(); ++y)
  { // For each line
    BYTE * pSrcPixel = pSrcLines[y];
    BYTE * pDstPixel = pDstLines[y];
    // fill with background (index 0) color
    memset(pDstPixel,0,DstLineLen);

    for (int x = 0; x < SrcLineLen; ++x)  // For each pixel
    {
      pRGBA = BPPSrc == 8 ? (BYTE*) &pPal[*pSrcPixel] : pSrcPixel;
      // the following criterion supposedly finds "dark" pixels; it may
      // need some twiddling and maybe use the alpha channel as well
      if (pRGBA[RGBA_RED] < 128 &&
          pRGBA[RGBA_GREEN] < 128 &&
          pRGBA[RGBA_BLUE] < 128 )
        pDstPixel[x / 8] |= 128 >> (x & 7);
      pSrcPixel += BPPSrc == 8 ? 1 : sizeof(RGBAPIXEL);
    }
  }
}

/*
void CBmp::quantize
    ( CBmp & rSrcBmp
    )
    // Performs color quantization on rSrcBmp. Assumes that memory
    // for the destination has been allocated already.
    // Written by Brett Porter, 981220
{
  COctree octree( 256 );

  ASSERT( GetBitsPerPixel() == 8 );
  ASSERT( rSrcBmp.GetBitsPerPixel() >= 16 );
  ASSERT( m_pClrTab );

  BYTE  **pSrcLines = rSrcBmp.GetLineArray();
  BYTE  **pDstLines = GetLineArray();
  int   x, y;

  // Add each pixel in the bitmap to the Octree

  for ( y = 0; y < GetHeight(); y++ )
  {
    RGBAPIXEL *pSrcPixel = ( RGBAPIXEL* )pSrcLines[y];

    for ( x = 0; x < GetWidth(); x++ )
    {
      octree.InsertColour( *pSrcPixel );
      pSrcPixel++;
    }
  }

  // Create the palette based on the Octree contents

  int nColours = 0;

  octree.MakePaletteTable( m_pClrTab, nColours );

  // Copy the bitmap, mapping the quantized colours as we go

  for ( y = 0; y < GetHeight(); y++ )
  {
    RGBAPIXEL *pSrcPixel = ( RGBAPIXEL* )pSrcLines[y];
    BYTE    *pDstPixel = pDstLines[y];

    for ( x = 0; x < GetWidth(); x++ )
    {
      *pDstPixel = octree.QuantizeColour( *pSrcPixel );
      pDstPixel++;
      pSrcPixel++;
    }
  }

  ASSERT_VALID(this);
}
*/

void CBmp::Empty()
{
	if (m_bEmpty)
		return;
  m_bEmpty = TRUE;
  freeMembers ();
  internalCreate(8, 8, 32, FALSE);
}
