/*
/--------------------------------------------------------------------
|
|      DIBSECT.CPP         Bitmap class using windows DIB sections.
|
|      This is a windows DIBSection wrapped in a CBmp-derived class.
|      It can be used just like a CWinBmp can be used. In addition,
|      CDIBSection can give access to the bitmap as a GDI bitmap handle.
|      This bitmap handle can be selected into a device context. All
|      normal GDI drawing functions can be used to write on the bitmap
|      in this way.
| 
|      Internally, CDIBSections are stored with header and bits in two 
|      separate buffers.
|
|      Copyright (c) 1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdafx.h"
#include "dibsect.h"
//#include "except.h"

#include <stdio.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC (CDIBSection, CWinBmp);


CDIBSection::CDIBSection
    ()
    // Creates an empty bitmap.
{
  // Delete everything the base class allocated.
  free(m_pBMI);
  m_pBMI = NULL;

  delete m_pLineArray;
  m_pLineArray = NULL;


  internalCreate(16, 16, 8, FALSE);

  ASSERT_VALID(this);
}


CDIBSection::~CDIBSection
    ()
{
  // Free the memory.
  freeMembers();
}


#ifdef _DEBUG
void CDIBSection::AssertValid
    () const
{
  // call inherited ASSERT_VALID first
  CWinBmp::AssertValid();

  // Bitmap must exist
  ASSERT (m_hBitmap);
}
#endif

void CDIBSection::Detach
    ()
{
  delete m_pLineArray;

  internalCreate(16, 16, 8, FALSE);

  ASSERT_VALID(this);
}

/////////////////////////////////////////////////////////////////////
// Windows-specific interface

void CDIBSection::Draw
    ( CDC * pDC,
      int x,
      int y,
      DWORD rop /* = SRCCOPY */
    )
    // Draw the DIB to a given DC.
{
  ASSERT_VALID (this);

	if (m_bEmpty)
		return;

  CDC SrcDC;
  SrcDC.CreateCompatibleDC (pDC);
  HGDIOBJ hOldBmp = ::SelectObject (SrcDC.GetSafeHdc(), m_hBitmap);
  if (m_pClrTab)
    BOOL bOK = ::SetDIBColorTable (SrcDC.GetSafeHdc(), 0, 256, (RGBQUAD *)m_pClrTab);


  ::BitBlt (pDC->GetSafeHdc(), 
            x, y, 
            GetWidth(),               // Destination width
            GetHeight(),              // Destination height
            SrcDC.GetSafeHdc(),
            0, 0,
            rop);
                  
  ::SelectObject (SrcDC.GetSafeHdc(), hOldBmp);
  SrcDC.DeleteDC();

}


void CDIBSection::DrawTransparent(CDC * pDC, int x, int y, COLORREF crColour)
{
    CDC dc,memDC, maskDC, tempDC;
	dc.Attach( pDC->GetSafeHdc() );
    maskDC.CreateCompatibleDC(&dc);
    CBitmap maskBitmap;
    
    //add these to store return of SelectObject() calls
    CBitmap* pOldMemBmp = NULL;
    CBitmap* pOldMaskBmp = NULL;
    HBITMAP hOldTempBmp = NULL;
    
    memDC.CreateCompatibleDC(&dc);
    tempDC.CreateCompatibleDC(&dc);
    CBitmap bmpImage;
	int nWidth = GetWidth();
	int nHeight = GetHeight();
    bmpImage.CreateCompatibleBitmap( &dc, nWidth, nHeight );
    pOldMemBmp = memDC.SelectObject( &bmpImage );
    
   
    hOldTempBmp = (HBITMAP) ::SelectObject( tempDC.GetSafeHdc(), m_hBitmap );
    
    memDC.BitBlt( 0,0,nWidth, nHeight, &tempDC, 0, 0, SRCCOPY );
    
    // Create monochrome bitmap for the mask
    maskBitmap.CreateBitmap( nWidth, nHeight, 1, 1, NULL );
    pOldMaskBmp = maskDC.SelectObject( &maskBitmap );
    memDC.SetBkColor( crColour );
    
    // Create the mask from the memory DC
    maskDC.BitBlt( 0, 0, nWidth, nHeight, &memDC, 
            0, 0, SRCCOPY );
    
    // Set the background in memDC to black. Using SRCPAINT with black 
    // and any other color results in the other color, thus making 
    // black the transparent color
    memDC.SetBkColor(RGB(0,0,0));
    memDC.SetTextColor(RGB(255,255,255));
    memDC.BitBlt(0, 0, nWidth, nHeight, &maskDC, 0, 0, SRCAND);
    
    // Set the foreground to black. See comment above.
    dc.SetBkColor(RGB(255,255,255));
    dc.SetTextColor(RGB(0,0,0));
    dc.BitBlt(x, y, nWidth, nHeight, &maskDC, 0, 0, SRCAND);
    
    // Combine the foreground with the background
    dc.BitBlt(x, y, nWidth, nHeight, &memDC, 
            0, 0, SRCPAINT);
    
    
    if (hOldTempBmp)
            ::SelectObject( tempDC.m_hDC, hOldTempBmp);
    if (pOldMaskBmp)
            maskDC.SelectObject( pOldMaskBmp );
    if (pOldMemBmp)
            memDC.SelectObject( pOldMemBmp );
    
    dc.Detach();
}

void CDIBSection::DrawAlpha
    ( CDC * pDC,
      int x,
      int y,
      BYTE Alpha 
    )
    // Draw the DIB to a given DC.
{
	if (m_bEmpty)
		return;

  ASSERT_VALID (this);

  //getfrom DC background
	CDIBSection backDIB;
	backDIB.FromCDC(pDC, x, y, GetWidth(), GetHeight());


	// Overlay picture
	int DestLineLen = GetWidth()*4;
	int SrcLineLen = backDIB.GetBytesPerLine();

	int maxy = GetHeight();
	int maxx =  GetWidth();

	int miny = 0;
	int minx = 0;

	int negalpha;

	BYTE alpha;
	for (int sy = miny; sy<maxy; sy++)
	{ // For each line
	  BYTE * pDest = backDIB.m_pBits+SrcLineLen*sy;
	  BYTE * pSrc1 = m_pBits+DestLineLen*sy;

	  for (int sx = minx; sx<maxx; sx++)
	  { // For each pixel
		  if (HasAlpha())
	          alpha = pSrc1[3];
		  else
			  alpha = Alpha;

		  negalpha = 255-alpha;
		  pDest[0] = (pDest[0]*negalpha+pSrc1[0]*Alpha)>>8;
		  pDest[1] = (pDest[1]*negalpha+pSrc1[1]*Alpha)>>8;
		  pDest[2] = (pDest[2]*negalpha+pSrc1[2]*Alpha)>>8;
		  pSrc1 += 4;
		  pDest += 4;
	  }
	}
 
	CDC SrcDC;
	SrcDC.CreateCompatibleDC (pDC);
	HGDIOBJ hOldBmp = ::SelectObject (SrcDC.GetSafeHdc(), backDIB.m_hBitmap);
	if (m_pClrTab)
	BOOL bOK = ::SetDIBColorTable (SrcDC.GetSafeHdc(), 0, 256, (RGBQUAD *)m_pClrTab);


	::BitBlt (pDC->GetSafeHdc(), 
			x, y, 
			GetWidth(),               // Destination width
			GetHeight(),              // Destination height
			SrcDC.GetSafeHdc(),
			0, 0,
			SRCCOPY);
          
	::SelectObject (SrcDC.GetSafeHdc(), hOldBmp);
	SrcDC.DeleteDC();

}



BOOL CDIBSection::DrawExtract
    ( CDC* pDC,
      CPoint pntDest,
      CRect rcSrc
    )
    // Draw part of the DIB on a DC.
{
  ASSERT_VALID (this);

  if (m_bEmpty)
		return FALSE;

  CDC SrcDC;
  SrcDC.CreateCompatibleDC (pDC);
  HGDIOBJ hOldBmp = ::SelectObject (SrcDC.GetSafeHdc(), m_hBitmap);
  
  BOOL bOK = ::BitBlt (pDC->GetSafeHdc (),
                       pntDest.x,         // Destination x
                       pntDest.y,         // Destination y
                       rcSrc.Width (),    // Source width
                       rcSrc.Height (),   // Source height
                       SrcDC.GetSafeHdc(),
                       rcSrc.left,        // Source x
                       rcSrc.top,
                       SRCCOPY);

  ::SelectObject (SrcDC.GetSafeHdc(), hOldBmp);
  SrcDC.DeleteDC();

  return bOK;
}

/////////////////////////////////////////////////////////////////////
// CDIBSection member access

HANDLE CDIBSection::GetHandle
    ()
    // Returns a GDI handle to the bitmap. This handle can be selected
    // into a DC and used in normal GDI operations. It must be released
    // with ReleaseHandle() before the bitmap pointer can be used again 
    // since GDI operations can be queued.
{
  return m_hBitmap;
}


/////////////////////////////////////////////////////////////////////
// Protected callbacks

void CDIBSection::internalCreate
    ( LONG Width,
      LONG Height,
      WORD BitsPerPixel,
      BOOL bAlphaChannel
    )
    // Create a new empty DIBSection.
    // Assumes that no memory is allocated before the call.
{
#ifdef MAX_BITMAP_SIZE
	if (GetMemNeeded (Width, Height, BitsPerPixel) 
      > MAX_BITMAP_SIZE)
	  throw CTextException(ERR_DIB_TOO_LARGE, "Bitmap size too large.\n");
#endif

  int MemNeeded = sizeof (BITMAPINFOHEADER);
  if (BitsPerPixel < 16)
  { // Color table exists
    MemNeeded += (1 << BitsPerPixel)*sizeof (RGBQUAD);
  }

  m_pBMI = (BITMAPINFOHEADER*) malloc (MemNeeded);

  // out of memory?
  if (!m_pBMI)
    AfxThrowMemoryException();

  // Fill in the header info.
  m_pBMI->biSize = sizeof(BITMAPINFOHEADER);
  m_pBMI->biWidth = Width;
  m_pBMI->biHeight = Height;
  m_pBMI->biPlanes = 1;
  m_pBMI->biBitCount = BitsPerPixel;
  m_pBMI->biCompression = BI_RGB;   // No compression
  m_pBMI->biSizeImage = 0;
  m_pBMI->biXPelsPerMeter = 0;
  m_pBMI->biYPelsPerMeter = 0;
  m_pBMI->biClrUsed = 0;           // Always use the whole palette.
  m_pBMI->biClrImportant = 0;

  // Create DIB section in shared memory
  m_hBitmap = CreateDIBSection (NULL, (BITMAPINFO *)m_pBMI, 
                                DIB_RGB_COLORS, (void **)&m_pBits, 0, 0l);
  if (!m_hBitmap)
    AfxThrowMemoryException();

  // Set color table pointer & pointer to bits.
  initPointers ();

  initLocals (Width, Height, BitsPerPixel, bAlphaChannel);

  ASSERT_VALID (this);
}


void CDIBSection::internalCreate
    ( BITMAPINFOHEADER* pBMI
    )
    // Creates a CDIBSection from an existing bitmap pointer.
    // Assumes that no memory is allocated before the call.
{
#ifdef MAX_BITMAP_SIZE
	if (GetMemNeeded (pBMI->biWidth, pBMI->biHeight, pBMI->biBitCount) 
      > MAX_BITMAP_SIZE)
	  throw CTextException(ERR_DIB_TOO_LARGE, "Bitmap size too large.\n");
#endif

  int MemNeeded = sizeof (BITMAPINFOHEADER);
  if (pBMI->biBitCount < 16)
  { // Color table exists
    MemNeeded += (1 << pBMI->biBitCount)*sizeof (RGBQUAD);
  }

  m_pBMI = (BITMAPINFOHEADER *) malloc (MemNeeded);

  memcpy (m_pBMI, pBMI, MemNeeded);

  // Create DIB section in shared memory
  m_hBitmap = CreateDIBSection (NULL, (BITMAPINFO *)m_pBMI, 
                                DIB_RGB_COLORS, (void **)&m_pBits, 0, 0l);
  if (!m_hBitmap)
    AfxThrowMemoryException();

  // Set color table pointer & pointer to bits.
  initPointers ();

  initLocals (m_pBMI->biWidth, m_pBMI->biHeight,
              m_pBMI->biBitCount, FALSE);

  ASSERT_VALID (this);
}


void CDIBSection::freeMembers
    ()
{
  delete m_pLineArray;
  m_pLineArray = NULL;

  free(m_pBMI);
  DeleteObject (m_hBitmap);

  m_pBMI = NULL;
}


HANDLE CDIBSection::createCopyHandle
    ()
    // Creates a copy of the current bitmap in a global memory block
    // and returns a handle to this block.
{
  HANDLE  hCopy;

  int MemUsed = GetMemUsed();

  hCopy = (HANDLE) ::GlobalAlloc (GMEM_MOVEABLE | GMEM_DDESHARE,
                                  MemUsed);
  if (hCopy == NULL)
    AfxThrowMemoryException();

  long * lpCopy = (long *) ::GlobalLock((HGLOBAL) hCopy);

  int BPP = GetBitsPerPixel();
  int HeaderMem = sizeof (BITMAPINFOHEADER);
  if (BPP < 16)
  { // Color table exists
    HeaderMem += (1 << BPP)*sizeof (RGBQUAD);
  }
  memcpy (lpCopy, m_pBMI, HeaderMem);
  int BitsMem = GetBitsMemNeeded (GetWidth(), GetHeight(), BPP);
  memcpy (((BYTE *)lpCopy)+HeaderMem, m_pBits, BPP);

  ::GlobalUnlock((HGLOBAL) hCopy);

  return hCopy;
}


void CDIBSection::initPointers
    ()
    // Set color table pointer based on m_pBMI. In this derived class,
    // m_pBits is set in internalCreate().
{
  if (m_pBMI->biBitCount < 16)
  { // Color table exists
    m_pClrTab = (RGBAPIXEL *)(((BITMAPINFO *) (m_pBMI))->bmiColors);
  }
  else
  { // No color table for 16 bpp and up.
    m_pClrTab = NULL;
  }
}

void CDIBSection::FromCDC(CDC * pDC, int x, int y, int nWidth, int nHeight)
{
	Create(nWidth,nHeight,32,FALSE);

	CDC SrcDC;
	SrcDC.CreateCompatibleDC (pDC);
	HGDIOBJ hOldBmp = ::SelectObject (SrcDC.GetSafeHdc(), GetHandle());

	SrcDC.BitBlt (0, 0, 
			GetWidth(),               // Destination width
			GetHeight(),              // Destination height
			pDC,
			x, y,
			SRCCOPY);
  
	::SelectObject (SrcDC.GetSafeHdc(), hOldBmp);
	SrcDC.DeleteDC();

	GdiFlush();
}
