/*
/--------------------------------------------------------------------
|
|      WINBMP.CPP         Device independent bitmap class
|
|        Windows version
|
|        Manipulates uncompressed device independent bitmaps
|        of all color depths.
|
|        Header and bits are in one buffer.
|        The bottom line is stored first. Color tables for 16-,
|        24-, and 32- bpp-bitmaps are not supported. biClrUsed is
|        always 0.
|
|        A CWinBmp can contain alpha channel information. As in TGA
|        files, 0 is completely transparent and 255 completely opaque.
|
|      Copyright (c) 1996-1998 Ulrich von Zadow
|
\--------------------------------------------------------------------
*/

#include "stdafx.h"
#include "winbmp.h"
//#include "except.h"
#include "windefs.h"

#include <stdio.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

IMPLEMENT_DYNAMIC (CWinBmp, CBmp);


CWinBmp::CWinBmp
    ()
    // Creates an empty bitmap.
{
  internalCreate(16, 16, 8, FALSE);

  ASSERT_VALID(this);
}


CWinBmp::~CWinBmp
    ()
{
  // Free the memory.
  freeMembers();
}


#ifdef _DEBUG
void CWinBmp::AssertValid
    () const
{
  // call inherited ASSERT_VALID first
  CBmp::AssertValid();

  // Bitmap must exist
  ASSERT (m_pBMI);

  // Bitmapinfo must equal member variables
  ASSERT (m_pBMI->biHeight == m_Height);
  ASSERT (m_pBMI->biWidth == m_Width);
  ASSERT (m_pBMI->biBitCount == m_bpp);

  // Only uncompressed bitmaps allowed.
  ASSERT (m_pBMI->biCompression == BI_RGB);

  // No optimized color tables allowed.
  ASSERT (m_pBMI->biClrUsed == 0 ||
          m_pBMI->biClrUsed == (DWORD)(1 << m_bpp));
}
#endif


/////////////////////////////////////////////////////////////////////
// CWinBmp manipulation

void CWinBmp::AlphaBlt
    ( CWinBmp * pSrcBmp,
      int x,
      int y
    )
    // Do a bitblt using the alpha channel of pSrcBmp.
    // Legacy routine. Should not be used.
{
  ASSERT_VALID (this);
  ASSERT (GetBitsPerPixel() == 32);

  ASSERT_VALID (pSrcBmp);

  // Overlay picture
  int DestLineLen = GetWidth()*4;
  int SrcLineLen = pSrcBmp->GetBytesPerLine();
  RGBAPIXEL * pPal = pSrcBmp->GetPalette();

  // Perform clipping.
  int maxy = min (pSrcBmp->GetHeight(),
                  GetHeight()-y);
  int maxx = min (pSrcBmp->GetWidth(),
                  GetWidth()-x);
  int miny = max (0,-y);
  int minx = max (0,-x);

  if (pSrcBmp->m_bAlphaChannel)
  {
    int alpha, negalpha;

    for (int sy = miny; sy<maxy; sy++)
    { // For each line
      BYTE * pDest = m_pBits+DestLineLen*(y+sy)+x*4;
      BYTE * pSrc = pSrcBmp->m_pBits+SrcLineLen*sy;

      for (int sx = minx; sx<maxx; sx++)
      { // For each pixel
        if (pPal)
        {
          RGBAPIXEL * pPixel = &(pPal[*pSrc]);
          alpha = pPixel[3];
          negalpha = 255-alpha;
          pDest[0] = (pDest[0]*negalpha+pPixel[0]*alpha)>>8;
          pDest[1] = (pDest[1]*negalpha+pPixel[1]*alpha)>>8;
          pDest[2] = (pDest[2]*negalpha+pPixel[2]*alpha)>>8;

          pSrc++;
        }
        else
        {
          alpha = pSrc[3];
          negalpha = 255-alpha;
          pDest[0] = (pDest[0]*negalpha+pSrc[0]*alpha)>>8;
          pDest[1] = (pDest[1]*negalpha+pSrc[1]*alpha)>>8;
          pDest[2] = (pDest[2]*negalpha+pSrc[2]*alpha)>>8;
          pSrc += 4;
        }

        pDest += 4;
      }
    }
  }
  else
  {
    for (int sy = miny; sy<maxy; sy++)
    { // For each line
      if (pPal)
      {
        BYTE * pDest = m_pBits+DestLineLen*(y+sy)+x*4;
        BYTE * pSrc = pSrcBmp->m_pBits+SrcLineLen*sy;

        for (int sx = minx; sx<maxx; sx++)
        { // For each pixel
          *((RGBAPIXEL *)pDest) = pPal[*pSrc];

          pDest += 4;
          pSrc++;
        }
      }
      else
      {
        BYTE * pDest = m_pBits+DestLineLen*(y+sy)+x*4;
        BYTE * pSrc = pSrcBmp->m_pBits+SrcLineLen*sy;

        memcpy (pDest, pSrc, 4*(maxx-minx));
      }
    }
  }
  ASSERT_VALID (this);
}



void CWinBmp::AlphaFade( CWinBmp * pSrcBmp1,CWinBmp * pSrcBmp2,BYTE Alpha)
{
	ASSERT_VALID (this);
	ASSERT (GetBitsPerPixel() == 32);
	ASSERT_VALID (pSrcBmp1);
	ASSERT_VALID (pSrcBmp2);

	ASSERT (pSrcBmp1->GetBitsPerPixel() == 32);
	ASSERT (pSrcBmp2->GetBitsPerPixel() == 32);


	// Overlay picture
	int DestLineLen = GetWidth()*4;
	int SrcLineLen = pSrcBmp1->GetBytesPerLine();
	int SrcLineLen2 = pSrcBmp2->GetBytesPerLine();

	// Perform clipping.
	int maxy = min (pSrcBmp1->GetHeight(),
				  GetHeight());
	int maxx = min (pSrcBmp1->GetWidth(),
				  GetWidth());
	int miny = 0;
	int minx = 0;

	int negalpha;

	for (int sy = miny; sy<maxy; sy++)
	{ // For each line
	  BYTE * pDest = m_pBits+DestLineLen*(sy);
	  BYTE * pSrc1 = pSrcBmp1->m_pBits+SrcLineLen*sy;
	  BYTE * pSrc2 = pSrcBmp2->m_pBits+SrcLineLen2*sy;

	  for (int sx = minx; sx<maxx; sx++)
	  { // For each pixel
		  negalpha = 255-Alpha;
		  pDest[0] = (pSrc1[0]*negalpha+pSrc2[0]*Alpha)>>8;
		  pDest[1] = (pSrc1[1]*negalpha+pSrc2[1]*Alpha)>>8;
		  pDest[2] = (pSrc1[2]*negalpha+pSrc2[2]*Alpha)>>8;
		  pSrc1 += 4;
		  pSrc2 += 4;

		  pDest += 4;
	  }
	}
 
  ASSERT_VALID (this);
}
/////////////////////////////////////////////////////////////////////
// CWinBmp information

long CWinBmp::GetMemUsed
    ()
    // Returns the memory used by the object.
{
  ASSERT_VALID (this);

  return GetMemNeeded (GetWidth(), GetHeight(), m_pBMI->biBitCount)+
         sizeof (*this);
}


long CWinBmp::GetBytesPerLine
    ()
    // Returns number of bytes used per line.
{
  // bdelmee code change
  int nBytes = m_Width*m_bpp / 8;
  if (m_bpp == 1 && m_Width % 8)
	  ++nBytes;
  // adjust to nearest DWORD-multiple
  return (nBytes + 3) & ~3;
}


/////////////////////////////////////////////////////////////////////
// Windows-specific interface


void CWinBmp::CreateRes
    ( int ID
    )
    // Loads a DIB from a resource. Fails if the bitmap is compressed.
{
  HRSRC  hRsrc;
  HGLOBAL hGlobal;
  BITMAPINFOHEADER * pBMI;

  ASSERT_VALID (this);

  hRsrc = FindResource(AfxGetInstanceHandle(),
                       MAKEINTRESOURCE (ID),
                       RT_BITMAP);

  ASSERT (hRsrc);  // Make sure resource exists.

  hGlobal = LoadResource(AfxGetInstanceHandle(), hRsrc);
  pBMI = (BITMAPINFOHEADER *) LockResource(hGlobal);

  // Delete any existing stuff.
  freeMembers ();

  // Copy data into local memory & init locals.
  internalCreate (pBMI);

  // Free global memory block.
  UnlockResource(hGlobal);
  FreeResource(hGlobal);

  ASSERT_VALID (this);
}



CSize CWinBmp::GetSize
    ()
    // Returns size in pixels
{
  ASSERT_VALID (this);

  return CSize(GetWidth(), GetHeight());
}


BITMAPINFOHEADER * CWinBmp::GetBMI
    ()
{
  ASSERT_VALID (this);

  return m_pBMI;
}

void CWinBmp::SaveAsBmp
    ( const char * pszFName
    )
{
  int BPP = GetBitsPerPixel();

  FILE * pFile = fopen (pszFName, "wb");
  if (pFile == NULL) return;

  BITMAPFILEHEADER BFH;
  BFH.bfType = *((WORD*)"BM");
  BFH.bfReserved1 = 0;
  BFH.bfReserved2 = 0;
  BFH.bfOffBits = sizeof (BITMAPFILEHEADER) +
                  sizeof (BITMAPINFOHEADER);
  if (BPP <= 8)   // include palette
    BFH.bfOffBits += (1 << BPP) * sizeof(RGBQUAD);

  BFH.bfSize = BFH.bfOffBits;
  if (BPP <= 8)
    BFH.bfSize += GetBytesPerLine()*m_Height;
  else
	  BFH.bfSize += (((m_Width * 3) + 3) & ~3)*m_Height;

  fwrite (&BFH, 1, sizeof (BITMAPFILEHEADER), pFile);

  if (BPP <= 8)
  {
    fwrite (m_pBMI, 1, sizeof (BITMAPINFOHEADER), pFile);
    fwrite (m_pClrTab, 1, (1 << BPP) * sizeof(RGBQUAD), pFile);
    fwrite (m_pBits, 1, GetBytesPerLine()*m_Height, pFile);
  }
  else
  {
    BITMAPINFOHEADER FileBMI = *m_pBMI;
    FileBMI.biBitCount = 24;  // not 32...
    fwrite (&FileBMI, 1, sizeof (BITMAPINFOHEADER), pFile);

    BYTE * pCurLine;
    int x,y;
    int LinePadding = 4-((m_Width*3)&3);
    if (LinePadding == 4)
      LinePadding = 0;
    for (y=m_Height-1; y>=0; y--)
    {
      pCurLine = m_pLineArray[y];
      for (x=0; x<m_Width; x++)
      {
        fwrite (pCurLine+x*4, 1, 3, pFile);
      }
      fwrite (" ", 1, LinePadding, pFile);
    }
  }
  fclose (pFile);
}


/////////////////////////////////////////////////////////////////////
// CWinBmp output

void CWinBmp::Draw
    ( CDC * pDC,
      int x,
      int y,
      DWORD rop /* = SRCCOPY */
    )
    // Draw the DIB to a given DC.
{
	if (m_bEmpty)
		return;

  ASSERT_VALID (this);

  ::StretchDIBits(pDC->GetSafeHdc(),
                  x,                        // Destination x
                  y,                        // Destination y
                  GetWidth(),               // Destination width
                  GetHeight(),              // Destination height
                  0,                        // Source x
                  0,                        // Source y
                  GetWidth(),               // Source width
                  GetHeight(),              // Source height
                  m_pBits,                  // Pointer to bits
                  (BITMAPINFO *) m_pBMI,    // BITMAPINFO
                  DIB_RGB_COLORS,           // Options
                  rop);                     // Raster operator code
}


void CWinBmp::StretchDraw
    ( CDC * pDC,
      int x,
      int y,
      double Factor,
      DWORD rop /* = SRCCOPY */
    )
    // Draw the DIB to a given DC.
{
	if (m_bEmpty)
		return;

  ASSERT_VALID (this);

  ::StretchDIBits(pDC->GetSafeHdc(),
                  x,                        // Destination x
                  y,                        // Destination y
                  int (Factor*GetWidth()),  // Destination width
                  int (Factor*GetHeight()), // Destination height
                  0,                        // Source x
                  0,                        // Source y
                  GetWidth(),               // Destination width
                  GetHeight(),              // Destination height
                  m_pBits,                  // Pointer to bits
                  (BITMAPINFO *) m_pBMI,    // BITMAPINFO
                  DIB_RGB_COLORS,           // Options
                  rop);                     // Raster operator code
}

void CWinBmp::DrawC
    ( CDC * pDC,
      int x,
      int y,
      int nWidth, int nHeight,
      DWORD rop /* = SRCCOPY */
    )
    // Draw the DIB to a given DC.
{
	if (m_bEmpty)
		return;

  ASSERT_VALID (this);

  ::StretchDIBits(pDC->GetSafeHdc(),
                  x,                        // Destination x
                  y,                        // Destination y
                  nWidth,  // Destination width
                  nHeight, // Destination height
                  0,                        // Source x
                  0,                        // Source y
                  GetWidth(),               // Destination width
                  GetHeight(),              // Destination height
                  m_pBits,                  // Pointer to bits
                  (BITMAPINFO *) m_pBMI,    // BITMAPINFO
                  DIB_RGB_COLORS,           // Options
                  rop);                     // Raster operator code
}

BOOL CWinBmp::DrawExtract
    ( CDC* pDC,
      CPoint pntDest,
      CRect rcSrc
    )
    // Draw part of the DIB on a DC.
{
	if (m_bEmpty)
		return FALSE;

  ASSERT_VALID (this);
  return ::SetDIBitsToDevice (pDC->GetSafeHdc (),
                              pntDest.x,         // Destination x
                              pntDest.y,         // Destination y
                              rcSrc.Width (),    // Source width
                              rcSrc.Height (),   // Source height
                              rcSrc.left,        // Source x
                              m_pBMI->biHeight - rcSrc.bottom,
                                                 // Source lower y
                              0,
                              m_pBMI->biHeight,
                              m_pBits,
                              (BITMAPINFO *) m_pBMI,
                              DIB_RGB_COLORS );
}


void CWinBmp::ToClipBoard
    ()
    // Puts a copy of the DIB in the clipboard
{
  ASSERT_VALID (this);

  if (::OpenClipboard(NULL))
  {
    EmptyClipboard();
    HANDLE hDIB = createCopyHandle ();
    SetClipboardData (CF_DIB, hDIB);
    CloseClipboard();
  }
}


/////////////////////////////////////////////////////////////////////
// Static functions

long CWinBmp::GetBitsMemNeeded
    ( LONG width,
      LONG height,
      WORD BitsPerPixel
    )
    // Returns memory needed by bitmap bits.
{
  // Calculate memory per line.
  int LineMem = width * BitsPerPixel / 8;

  // bdelmee code change
  if (BitsPerPixel == 1 && width % 8)
	  ++LineMem;

  // Multiply by number of (DWORD-aligned) lines
  return height * ((LineMem + 3) & ~3);
}


long CWinBmp::GetMemNeeded
    ( LONG width,
      LONG height,
      WORD BitsPerPixel
    )
    // Returns memory needed by a bitmap with the specified attributes.
{
  int HeaderMem = sizeof(BITMAPINFOHEADER); // Header memory
  if (BitsPerPixel < 16)
  { // Palette memory
    HeaderMem += (1 << BitsPerPixel)*sizeof (RGBQUAD);
  }

  return HeaderMem+GetBitsMemNeeded (width, height, BitsPerPixel);
}


/////////////////////////////////////////////////////////////////////
// Protected callbacks

void CWinBmp::internalCreate
    ( LONG Width,
      LONG Height,
      WORD BitsPerPixel,
      BOOL bAlphaChannel
    )
    // Create a new empty DIB. Bits are uninitialized.
    // Assumes that no memory is allocated before the call.
{
  // Allocate memory
  int MemNeeded = GetMemNeeded (Width, Height, BitsPerPixel);

#ifdef MAX_BITMAP_SIZE
  if (MemNeeded > MAX_BITMAP_SIZE)
    throw CTextException(ERR_DIB_TOO_LARGE, "Bitmap size too large.\n");
#endif

  m_pBMI = (BITMAPINFOHEADER*) malloc (MemNeeded);

  // out of memory?
  if (!m_pBMI)
    AfxThrowMemoryException();

  // Fill in the header info.
  m_pBMI->biSize = sizeof(BITMAPINFOHEADER);
  m_pBMI->biWidth = Width;
  m_pBMI->biHeight = Height;
  m_pBMI->biPlanes = 1;
  m_pBMI->biBitCount = BitsPerPixel;
  m_pBMI->biCompression = BI_RGB;   // No compression
  m_pBMI->biSizeImage = 0;
  m_pBMI->biXPelsPerMeter = 0;
  m_pBMI->biYPelsPerMeter = 0;
  m_pBMI->biClrUsed = 0;           // Always use the whole palette.
  m_pBMI->biClrImportant = 0;

  // Set color table pointer & pointer to bits.
  initPointers ();

  initLocals (Width, Height, BitsPerPixel, bAlphaChannel);

  ASSERT_VALID (this);
}


void CWinBmp::internalCreate
    ( BITMAPINFOHEADER* pBMI
    )
    // Creates a CWinBmp from an existing bitmap pointer.
    // Assumes that no memory is allocated before the call.
{
	int MemNeeded = GetMemNeeded (pBMI->biWidth, pBMI->biHeight,
								  pBMI->biBitCount);
#ifdef MAX_BITMAP_SIZE
	if (MemNeeded > MAX_BITMAP_SIZE)
	  throw CTextException(ERR_DIB_TOO_LARGE, "Bitmap size too large.\n");
#endif

	m_pBMI = (BITMAPINFOHEADER *) malloc (MemNeeded);

	// out of memory?
	if (!m_pBMI)
	  AfxThrowMemoryException();

	memcpy (m_pBMI, pBMI, MemNeeded);

  // Set color table pointer & pointer to bits.
  initPointers ();

  initLocals (m_pBMI->biWidth, m_pBMI->biHeight,
              m_pBMI->biBitCount, FALSE);

  ASSERT_VALID (this);
}


void CWinBmp::freeMembers
    ()
{
  free(m_pBMI);
  m_pBMI = NULL;

  delete m_pLineArray;
  m_pLineArray = NULL;
}


void CWinBmp::initLineArray
    ()
{
  m_pLineArray = new BYTE * [m_Height];
  int LineLen = GetBytesPerLine();

  for (int y=0; y<m_Height; y++)
    m_pLineArray[y] = m_pBits + (m_Height-y-1)*LineLen;
}


HANDLE CWinBmp::createCopyHandle
    ()
    // Creates a copy of the current bitmap in a global memory block
    // and returns a handle to this block.
{
  HANDLE  hCopy;

  int MemUsed = GetMemUsed();

  hCopy = (HANDLE) ::GlobalAlloc (GMEM_MOVEABLE | GMEM_DDESHARE,
                                  MemUsed);
  if (hCopy == NULL)
    AfxThrowMemoryException();

  long * lpCopy = (long *) ::GlobalLock((HGLOBAL) hCopy);
  memcpy (lpCopy, m_pBMI, MemUsed);

  ::GlobalUnlock((HGLOBAL) hCopy);

  return hCopy;
}


void CWinBmp::initPointers
    ()
    // Set color table pointer & pointer to bits based on m_pBMI.
{
  if (m_pBMI->biBitCount < 16)
  { // Color table exists
    m_pClrTab = (RGBAPIXEL *)(((BITMAPINFO *) (m_pBMI))->bmiColors);

    m_pBits = (BYTE *)m_pClrTab +
              (1 << m_pBMI->biBitCount)*sizeof (RGBQUAD);
  }
  else
  { // No color table for 16 bpp and up.
    m_pClrTab = NULL;
    m_pBits =  (BYTE *)(((BITMAPINFO *) (m_pBMI))->bmiColors);
  }
}

void CWinBmp::FillAlphaRGB(BYTE * pBits, int nWidth, int nHeight)
{

  m_bAlphaChannel = TRUE;

  int y;                            // Current row
  int x;                            // Current column

  BYTE * pSrcLine = pBits;          // Start of current row in file.
  BYTE * pDest;                     // Current destination.
  BYTE * pSrc;                      // Current position in file.
  int    XSize = GetWidth();// Width of bitmap in pixels.
  int    LineLen = nWidth*3;
                                    // Width of source in bytes
                                    //   (DWORD-aligned).
  BYTE ** pLineArray = GetLineArray();
                                   // Pointers to dest. lines.


  for (y=0; y<GetHeight(); y++)
  { // For each line...

	pDest = pLineArray[y];
    pSrc = pSrcLine;

    for (x=0; x<XSize; x++)
    { // For each pixel...
      //*(pDest+RGBA_RED) = ((WINRGBQUAD *)pSrc)->rgbBlue;
      //*(pDest+RGBA_GREEN) = ((WINRGBQUAD *)pSrc)->rgbGreen;
      //*(pDest+RGBA_BLUE) = ((WINRGBQUAD *)pSrc)->rgbRed;
      *(pDest+RGBA_ALPHA) = ((WINRGBQUAD *)pSrc)->rgbRed;
      pDest += 4;
      pSrc += 3;
    }
    pSrcLine += LineLen;
  }

}

void CWinBmp::FromRGB(BYTE * pBits, int nWidth, int nHeight)
{
  Create (nWidth, nHeight, 32, FALSE);

  int y;                            // Current row
  int x;                            // Current column

  BYTE * pSrcLine = pBits;          // Start of current row in file.
  BYTE * pDest;                     // Current destination.
  BYTE * pSrc;                      // Current position in file.
  int    XSize = GetWidth();// Width of bitmap in pixels.
  int    LineLen = nWidth*3;
                                    // Width of source in bytes
                                    //   (DWORD-aligned).
  BYTE ** pLineArray = GetLineArray();
                                   // Pointers to dest. lines.

  //Trace (2, "Decoding 24 bit per pixel bitmap.\n");

  for (y=0; y<GetHeight(); y++)
  { // For each line...
    //pDest = pLineArray[GetHeight()-y-1];
	pDest = pLineArray[y];
    pSrc = pSrcLine;
    for (x=0; x<XSize; x++)
    { // For each pixel...
      *(pDest+RGBA_RED) = ((WINRGBQUAD *)pSrc)->rgbBlue;
      *(pDest+RGBA_GREEN) = ((WINRGBQUAD *)pSrc)->rgbGreen;
      *(pDest+RGBA_BLUE) = ((WINRGBQUAD *)pSrc)->rgbRed;
      *(pDest+RGBA_ALPHA) = 0xFF;
      pDest += 4;
      pSrc += 3;
    }
    pSrcLine += LineLen;
  }

}

DWORD CWinBmp::BitsSize()
{
	if (GetBitsPerPixel()==32)
		return GetWidth()*GetHeight()*4;

	if (GetBitsPerPixel()==24)
		return GetWidth()*GetHeight()*3;

	return 0;
}

BYTE* CWinBmp::GetBits()
{
	if (GetBitsPerPixel()==32)
		return m_pBits;
	else
		return NULL;
}


void CWinBmp::ToRGB(BYTE * pBits)
{
  int y;                            // Current row
  int x;                            // Current column

  BYTE * pSrcLine = pBits;          // Start of current row in file.
  BYTE * pDest;                     // Current destination.
  BYTE * pSrc;                      // Current position in file.
  int    XSize = GetWidth();// Width of bitmap in pixels.
  int    LineLen = GetWidth()*3;
                                    // Width of source in bytes
                                    //   (DWORD-aligned).
  BYTE ** pLineArray = GetLineArray();
                                   // Pointers to dest. lines.


  for (y=0; y<GetHeight(); y++)
  { // For each line...
    //pDest = pLineArray[GetHeight()-y-1];
	pDest = pLineArray[y];
    pSrc = pSrcLine;

    for (x=0; x<XSize; x++)
    { // For each pixel...
      ((WINRGBQUAD *)pSrc)->rgbBlue = *(pDest+RGBA_RED);
      ((WINRGBQUAD *)pSrc)->rgbGreen = *(pDest+RGBA_GREEN) ;
      ((WINRGBQUAD *)pSrc)->rgbRed =*(pDest+RGBA_BLUE); 

      pDest += 4;
      pSrc += 3;
    }
    pSrcLine += LineLen;
  }
}

void CWinBmp::AlphaToRGB(BYTE * pBits)
{
  int y;                            // Current row
  int x;                            // Current column

  BYTE * pSrcLine = pBits;          // Start of current row in file.
  BYTE * pDest;                     // Current destination.
  BYTE * pSrc;                      // Current position in file.
  int    XSize = GetWidth();// Width of bitmap in pixels.
  int    LineLen = GetWidth()*3;
                                    // Width of source in bytes
                                    //   (DWORD-aligned).
  BYTE ** pLineArray = GetLineArray();
                                   // Pointers to dest. lines.


  for (y=0; y<GetHeight(); y++)
  { // For each line...
    //pDest = pLineArray[GetHeight()-y-1];
	pDest = pLineArray[y];
    pSrc = pSrcLine;

    for (x=0; x<XSize; x++)
    { // For each pixel...
      ((WINRGBQUAD *)pSrc)->rgbBlue = *(pDest+RGBA_ALPHA);
      ((WINRGBQUAD *)pSrc)->rgbGreen = *(pDest+RGBA_ALPHA) ;
      ((WINRGBQUAD *)pSrc)->rgbRed =*(pDest+RGBA_ALPHA); 

      pDest += 4;
      pSrc += 3;
    }
    pSrcLine += LineLen;
  }
}

DWORD CWinBmp::GetSizeForRGB()
{
	return GetWidth()*GetHeight()*3;
}
